// API para gestionar los servicios
const API = {
    // Clave para almacenar los datos en localStorage
    STORAGE_KEY: 'classifieds_data',
    
    // Usuario actual (simulado)
    currentUser: {
      id: 999,
      name: "Usuario Actual",
      avatar: "placeholder.svg"
    },
    
    // Obtener todos los clasificados
    getAllClassifieds() {
      const storedData = localStorage.getItem(this.STORAGE_KEY);
      if (storedData) {
        const parsedData = JSON.parse(storedData);
        // Convertir las fechas de string a objetos Date
        return parsedData.map(item => ({
          ...item,
          createdAt: new Date(item.createdAt)
        }));
      }
      
      // Si no hay datos almacenados, usar datos de ejemplo
      return this.initializeData();
    },
    
    // Inicializar datos de ejemplo
    initializeData() {
      const initialData = [
        {
          id: 1,
          title: "Diseño de logotipo profesional para tu marca",
          description: "Crearé un logotipo único y profesional que represente la identidad de tu marca. Incluye revisiones ilimitadas y archivos en alta resolución.",
          price: "$120",
          category: "Diseño gráfico",
          rating: 4.9,
          reviews: 156,
          deliveryTime: "3 días",
          location: "Madrid, España",
          provider: {
            id: 101,
            name: "Laura Martínez",
            avatar: "placeholder.svg"
          },
          createdAt: new Date(2023, 11, 15),
          pinned: false,
          isNew: false,
          featured: true,
          type: "offer"
        },
        {
          id: 2,
          title: "Desarrollo de sitio web con WordPress",
          description: "Crearé un sitio web profesional y responsivo utilizando WordPress. Incluye instalación, configuración y personalización de tema.",
          price: "$350",
          category: "Desarrollo web",
          rating: 4.8,
          reviews: 89,
          deliveryTime: "7 días",
          location: "Barcelona, España",
          provider: {
            id: 102,
            name: "Carlos Rodríguez",
            avatar: "placeholder.svg"
          },
          createdAt: new Date(2023, 11, 20),
          pinned: false,
          isNew: false,
          featured: false,
          type: "offer"
        },
        {
          id: 3,
          title: "Necesito traductor inglés-español para documentos técnicos",
          description: "Busco un traductor profesional para traducir manuales técnicos del inglés al español. Se requiere experiencia en terminología técnica.",
          price: "$0.05/palabra",
          category: "Traducción",
          rating: 4.7,
          reviews: 124,
          deliveryTime: "24 horas",
          location: "Valencia, España",
          provider: {
            id: 103,
            name: "Ana García",
            avatar: "placeholder.svg"
          },
          createdAt: new Date(2023, 11, 25),
          pinned: true,
          isNew: false,
          featured: true,
          type: "request"
        },
        {
          id: 4,
          title: "Busco especialista en marketing para redes sociales",
          description: "Necesito un experto en marketing digital para desarrollar una estrategia completa para mis redes sociales. Incluye análisis de competencia y calendario editorial.",
          price: "$200",
          category: "Marketing digital",
          rating: 4.6,
          reviews: 78,
          deliveryTime: "5 días",
          location: "Sevilla, España",
          provider: {
            id: 104,
            name: "Miguel Fernández",
            avatar: "placeholder.svg"
          },
          createdAt: new Date(2023, 12, 1),
          pinned: false,
          isNew: false,
          featured: false,
          type: "request"
        },
        {
          id: 5,
          title: "Edición y retoque de fotografías profesionales",
          description: "Servicio de edición y retoque fotográfico profesional. Corrección de color, retoque de piel, eliminación de fondos y más.",
          price: "$40/foto",
          category: "Diseño gráfico",
          rating: 4.9,
          reviews: 112,
          deliveryTime: "2 días",
          location: "Málaga, España",
          provider: {
            id: 105,
            name: "Sofía López",
            avatar: "placeholder.svg"
          },
          createdAt: new Date(2023, 12, 5),
          pinned: false,
          isNew: true,
          featured: false,
          type: "offer"
        },
        {
          id: 6,
          title: "Necesito desarrollador para aplicación móvil",
          description: "Busco desarrollador con experiencia en aplicaciones móviles nativas para iOS y Android. El proyecto incluye diseño de UI/UX, desarrollo y publicación en tiendas.",
          price: "$1500",
          category: "Programación",
          rating: 4.8,
          reviews: 67,
          deliveryTime: "30 días",
          location: "Bilbao, España",
          provider: {
            id: 999, // Este es del usuario actual
            name: "Usuario Actual",
            avatar: "placeholder.svg"
          },
          createdAt: new Date(2023, 12, 10),
          pinned: false,
          isNew: true,
          featured: false,
          type: "request"
        }
      ];
      
      // Guardar en localStorage
      localStorage.setItem(this.STORAGE_KEY, JSON.stringify(initialData));
      
      return initialData;
    },
    
    // Obtener clasificados filtrados
    getFilteredClassifieds(type, filter) {
      let classifieds = this.getAllClassifieds();
      
      // Filtrar por tipo de servicio
      if (type) {
        classifieds = classifieds.filter(classified => classified.type === type);
      }
      
      // Aplicar filtros adicionales
      if (filter === "recent") {
        const oneWeekAgo = new Date();
        oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
        classifieds = classifieds.filter(classified => classified.createdAt >= oneWeekAgo);
      } else if (filter === "popular") {
        classifieds = classifieds.filter(classified => classified.rating >= 4.7);
      } else if (filter === "featured") {
        classifieds = classifieds.filter(classified => classified.featured);
      } else if (filter && filter !== "all") {
        // Filtrar por categoría
        classifieds = classifieds.filter(classified => 
          classified.category.toLowerCase().includes(filter.toLowerCase())
        );
      }
      
      // Ordenar: primero anclados, luego nuevos, luego por fecha
      return classifieds.sort((a, b) => {
        if (a.pinned && !b.pinned) return -1;
        if (!a.pinned && b.pinned) return 1;
        
        const aIsNew = a.isNew || (new Date() - a.createdAt) < 3 * 24 * 60 * 60 * 1000;
        const bIsNew = b.isNew || (new Date() - b.createdAt) < 3 * 24 * 60 * 60 * 1000;
        
        if (aIsNew && !bIsNew) return -1;
        if (!aIsNew && bIsNew) return 1;
        
        return b.createdAt - a.createdAt;
      });
    },
    
    // Buscar clasificados por texto
    searchClassifieds(query, type) {
      if (!query) return this.getFilteredClassifieds(type);
      
      const classifieds = this.getFilteredClassifieds(type);
      const searchQuery = query.toLowerCase();
      
      return classifieds.filter(classified => 
        classified.title.toLowerCase().includes(searchQuery) ||
        classified.description.toLowerCase().includes(searchQuery) ||
        classified.category.toLowerCase().includes(searchQuery) ||
        classified.location.toLowerCase().includes(searchQuery)
      );
    },
    
    // Obtener un clasificado por ID
    getClassifiedById(id) {
      const classifieds = this.getAllClassifieds();
      return classifieds.find(classified => classified.id === id);
    },
    
    // Crear un nuevo clasificado
    createClassified(classifiedData) {
      const classifieds = this.getAllClassifieds();
      
      // Generar un nuevo ID
      const newId = classifieds.length > 0 
        ? Math.max(...classifieds.map(c => c.id)) + 1 
        : 1;
      
      // Crear el nuevo clasificado
      const newClassified = {
        id: newId,
        ...classifiedData,
        provider: this.currentUser,
        createdAt: new Date(),
        pinned: false,
        isNew: true,
        featured: false,
        rating: 5.0,
        reviews: 0
      };
      
      // Añadir a la lista y guardar
      classifieds.push(newClassified);
      localStorage.setItem(this.STORAGE_KEY, JSON.stringify(classifieds));
      
      return newClassified;
    },
    
    // Actualizar un clasificado
    updateClassified(id, updates) {
      const classifieds = this.getAllClassifieds();
      const index = classifieds.findIndex(c => c.id === id);
      
      if (index !== -1) {
        // Actualizar solo los campos proporcionados
        classifieds[index] = { ...classifieds[index], ...updates };
        localStorage.setItem(this.STORAGE_KEY, JSON.stringify(classifieds));
        return classifieds[index];
      }
      
      return null;
    },
    
    // Eliminar un clasificado
    deleteClassified(id) {
      const classifieds = this.getAllClassifieds();
      const index = classifieds.findIndex(c => c.id === id);
      
      if (index !== -1) {
        classifieds.splice(index, 1);
        localStorage.setItem(this.STORAGE_KEY, JSON.stringify(classifieds));
        return true;
      }
      
      return false;
    },
    
    // Anclar/desanclar un clasificado
    togglePinClassified(id) {
      const classified = this.getClassifiedById(id);
      
      if (classified) {
        return this.updateClassified(id, { pinned: !classified.pinned });
      }
      
      return null;
    }
  };
  
  // Controlador de la aplicación
  const App = {
    // Estado de la aplicación
    state: {
      currentServiceType: "offer",
      currentFilter: "all",
      searchQuery: ""
    },
    
    // Inicializar la aplicación
    init() {
      this.renderClassifieds();
      this.setupEventListeners();
      
      // Mostrar iniciales del usuario actual en el avatar
      const currentUserAvatar = document.getElementById('current-user-avatar');
      currentUserAvatar.textContent = this.getInitials(API.currentUser.name);
    },
    
    // Obtener iniciales de un nombre
    getInitials(name) {
      return name
        .split(' ')
        .map(word => word[0])
        .join('')
        .toUpperCase();
    },
    
    // Formatear fecha
    formatDate(date) {
      const now = new Date();
      const diffTime = Math.abs(now - date);
      const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
      
      if (diffDays === 0) {
        return "Hoy";
      } else if (diffDays === 1) {
        return "Ayer";
      } else if (diffDays < 7) {
        return `Hace ${diffDays} días`;
      } else {
        return date.toLocaleDateString();
      }
    },
    
    // Generar estrellas de valoración
    generateRatingStars(rating) {
      const fullStars = Math.floor(rating);
      const halfStar = rating % 1 >= 0.5;
      const emptyStars = 5 - fullStars - (halfStar ? 1 : 0);
      
      let starsHTML = '';
      
      // Estrellas completas
      for (let i = 0; i < fullStars; i++) {
        starsHTML += '<i class="fas fa-star"></i>';
      }
      
      // Media estrella si corresponde
      if (halfStar) {
        starsHTML += '<i class="fas fa-star-half-alt"></i>';
      }
      
      // Estrellas vacías
      for (let i = 0; i < emptyStars; i++) {
        starsHTML += '<i class="far fa-star"></i>';
      }
      
      return starsHTML;
    },
    
    // Obtener etiqueta de calidad según valoración
    getRatingBadge(rating) {
      if (rating >= 4.5) return '<span class="rating-badge high">Excelente</span>';
      if (rating >= 4.0) return '<span class="rating-badge medium">Muy bueno</span>';
      if (rating >= 3.0) return '<span class="rating-badge">Bueno</span>';
      return '<span class="rating-badge low">Regular</span>';
    },
    
    // Renderizar clasificados
    renderClassifieds() {
      const classifiedWall = document.getElementById('classified-wall');
      classifiedWall.innerHTML = '';
      
      // Obtener clasificados filtrados
      let filteredClassifieds;
      
      if (this.state.searchQuery) {
        filteredClassifieds = API.searchClassifieds(
          this.state.searchQuery, 
          this.state.currentServiceType
        );
      } else {
        filteredClassifieds = API.getFilteredClassifieds(
          this.state.currentServiceType, 
          this.state.currentFilter
        );
      }
      
      // Añadir botón para crear clasificado
      const addClassifiedBtn = document.createElement('div');
      addClassifiedBtn.className = 'add-classified-btn';
      addClassifiedBtn.innerHTML = `
        <i class="fas fa-plus-circle"></i>
        <span>Publicar nuevo clasificado</span>
      `;
      addClassifiedBtn.addEventListener('click', () => this.showCreateClassifiedModal());
      classifiedWall.appendChild(addClassifiedBtn);
      
      // Renderizar clasificados
      filteredClassifieds.forEach(classified => {
        const cardEl = this.createClassifiedCard(classified);
        classifiedWall.appendChild(cardEl);
      });
      
      // Mensaje si no hay clasificados
      if (filteredClassifieds.length === 0) {
        const emptyMessage = document.createElement('div');
        emptyMessage.className = 'empty-message';
        emptyMessage.textContent = 'No hay clasificados disponibles con los filtros actuales.';
        emptyMessage.style.gridColumn = '1 / -1';
        emptyMessage.style.textAlign = 'center';
        emptyMessage.style.padding = '2rem';
        emptyMessage.style.color = '#666';
        classifiedWall.appendChild(emptyMessage);
      }
    },
    
    // Crear tarjeta de clasificado
    createClassifiedCard(classified) {
      const cardEl = document.createElement('div');
      cardEl.className = `classified-card ${classified.type}`;
      cardEl.dataset.classifiedId = classified.id;
      cardEl.style.animationDelay = `${Math.random() * 0.3}s`;
  
      // Verificar si el clasificado es del usuario actual
      const isOwner = classified.provider.id === API.currentUser.id;
  
      // Contenido principal de la tarjeta
      let cardHTML = `
        <div class="card-content">
          <div class="card-header">
            <div class="card-type ${classified.type}">
              ${classified.type === 'offer' ? 'Oferta' : 'Solicitud'}
            </div>
            <div class="card-price">${classified.price}</div>
          </div>
          <h3 class="card-title">${classified.title}</h3>
          
          <!-- Sistema de valoración que reemplaza la imagen -->
          <div class="service-rating">
            <div class="rating-stars">
              ${this.generateRatingStars(classified.rating)}
            </div>
            <div class="rating-info">
              <span class="rating-number">${classified.rating}</span>
              <span class="rating-count">(${classified.reviews} reseñas)</span>
              ${this.getRatingBadge(classified.rating)}
            </div>
            <div class="click-indicator">
              <i class="fas fa-mouse-pointer"></i> Click para más detalles
            </div>
          </div>
          
          <div class="card-category">
            <i class="fas fa-tag"></i>
            <span>${classified.category}</span>
          </div>
          <p class="card-description">${classified.description}</p>
          <div class="card-meta">
            <div class="card-meta-item">
              <i class="fas fa-clock"></i>
              <span>${classified.deliveryTime}</span>
            </div>
            <div class="card-meta-item">
              <i class="fas fa-map-marker-alt"></i>
              <span>${classified.location}</span>
            </div>
          </div>
          <div class="card-footer">
            <div class="card-provider">
              <div class="provider-avatar">${this.getInitials(classified.provider.name)}</div>
              <div class="provider-info">
                <span class="provider-name">${classified.provider.name}</span>
                <span class="provider-date">${this.formatDate(classified.createdAt)}</span>
              </div>
            </div>
            <div class="card-actions">
              <button class="card-action-btn contact-btn" data-classified-id="${classified.id}">
                <i class="fas fa-${classified.type === 'offer' ? 'shopping-cart' : 'hand-holding-usd'}"></i>
              </button>
            </div>
          </div>
        </div>
      `;
  
      // Añadir botón de anclar/desanclar
      cardHTML += `
        <button class="pin-btn ${classified.pinned ? 'pinned' : ''}" data-classified-id="${classified.id}">
          <i class="fas fa-thumbtack"></i>
        </button>
      `;
  
      // Añadir botón de eliminar solo si es el propietario
      if (isOwner) {
        cardHTML += `
          <button class="delete-btn" data-classified-id="${classified.id}">
            <i class="fas fa-trash-alt"></i>
          </button>
        `;
      }
  
      // Añadir etiqueta de "Nuevo" si es reciente (menos de 3 días)
      const threeDaysAgo = new Date();
      threeDaysAgo.setDate(threeDaysAgo.getDate() - 3);
      
      if (classified.isNew || classified.createdAt > threeDaysAgo) {
        cardHTML += `<div class="new-badge">Nuevo</div>`;
      }
  
      // Añadir etiqueta de "Destacado" si corresponde
      if (classified.featured) {
        cardHTML += `<div class="featured-badge">Destacado</div>`;
      }
  
      cardEl.innerHTML = cardHTML;
  
      // Configurar evento para contactar
      cardEl.querySelector('.contact-btn').addEventListener('click', (e) => {
        e.stopPropagation(); // Evitar que se propague al hacer clic en el botón
        this.showContactClassifiedModal(classified);
      });
  
      // Configurar evento para anclar/desanclar
      const pinBtn = cardEl.querySelector('.pin-btn');
      if (pinBtn) {
        pinBtn.addEventListener('click', (e) => {
          e.stopPropagation();
          this.togglePinClassified(classified.id);
        });
      }
  
      // Configurar evento para eliminar (solo si es el propietario)
      const deleteBtn = cardEl.querySelector('.delete-btn');
      if (deleteBtn) {
        deleteBtn.addEventListener('click', (e) => {
          e.stopPropagation();
          this.deleteClassified(classified.id);
        });
      }
  
      // Añadir evento de clic a toda la tarjeta para expandir/contraer en móviles
      cardEl.addEventListener('click', function() {
        // En dispositivos móviles, podríamos añadir una clase para expandir
        if (window.innerWidth <= 768) {
          this.classList.toggle('expanded');
        }
      });
  
      return cardEl;
    },
    
    // Anclar/desanclar clasificado
    togglePinClassified(id) {
      API.togglePinClassified(id);
      this.renderClassifieds();
    },
    
    // Eliminar clasificado
    deleteClassified(id) {
      if (confirm('¿Estás seguro de que deseas eliminar este clasificado?')) {
        API.deleteClassified(id);
        this.renderClassifieds();
      }
    },
    
    // Mostrar modal para crear clasificado
    showCreateClassifiedModal() {
      const createClassifiedModal = document.getElementById('create-classified-modal');
      createClassifiedModal.classList.add('active');
      
      // Actualizar título según el tipo de servicio
      const modalTitle = createClassifiedModal.querySelector('.modal-title');
      const modalDescription = createClassifiedModal.querySelector('.modal-description');
      const submitButton = document.getElementById('submit-classified-btn');
      
      if (this.state.currentServiceType === 'offer') {
        modalTitle.textContent = 'Publicar una oferta de servicio';
        modalDescription.textContent = 'Completa el formulario para ofrecer tus servicios a potenciales clientes.';
        submitButton.textContent = 'Publicar oferta';
      } else {
        modalTitle.textContent = 'Publicar una solicitud de servicio';
        modalDescription.textContent = 'Completa el formulario para publicar tu solicitud de servicio.';
        submitButton.textContent = 'Publicar solicitud';
      }
      
      // Limpiar formulario
      document.getElementById('create-classified-form').reset();
    },
    
    // Ocultar modal para crear clasificado
    hideCreateClassifiedModal() {
      const createClassifiedModal = document.getElementById('create-classified-modal');
      createClassifiedModal.classList.remove('active');
    },
    
    // Crear nuevo clasificado
    createClassified() {
      const title = document.getElementById('classified-title').value.trim();
      const category = document.getElementById('classified-category').value;
      const price = document.getElementById('classified-price').value;
      const description = document.getElementById('classified-description').value.trim();
      const deliveryTime = document.getElementById('classified-delivery').value;
      const location = document.getElementById('classified-location').value.trim();
      
      if (title && category && price && description && deliveryTime && location) {
        // Crear nuevo clasificado
        API.createClassified({
          title,
          description,
          price: `$${price}`,
          category,
          deliveryTime,
          location,
          type: this.state.currentServiceType
        });
        
        this.renderClassifieds();
        this.hideCreateClassifiedModal();
      } else {
        alert('Por favor, completa todos los campos obligatorios.');
      }
    },
    
    // Mostrar modal para contactar por clasificado
    showContactClassifiedModal(classified) {
      const contactClassifiedModal = document.getElementById('contact-classified-modal');
      const contactClassifiedTitle = document.getElementById('contact-classified-title');
      
      contactClassifiedModal.classList.add('active');
      contactClassifiedTitle.textContent = classified.title;
      contactClassifiedModal.dataset.classifiedId = classified.id;
      
      // Actualizar título según el tipo de servicio
      const modalTitle = contactClassifiedModal.querySelector('.modal-title');
      const submitButton = document.getElementById('submit-contact-btn');
      
      if (classified.type === 'offer') {
        modalTitle.textContent = 'Contratar este servicio';
        submitButton.textContent = 'Contratar servicio';
      } else {
        modalTitle.textContent = 'Aplicar a esta solicitud';
        submitButton.textContent = 'Enviar propuesta';
      }
      
      // Limpiar formulario
      document.getElementById('contact-classified-form').reset();
    },
    
    // Ocultar modal para contactar
    hideContactClassifiedModal() {
      const contactClassifiedModal = document.getElementById('contact-classified-modal');
      contactClassifiedModal.classList.remove('active');
    },
    
    // Enviar contacto
    submitContact() {
      const contactClassifiedModal = document.getElementById('contact-classified-modal');
      const name = document.getElementById('contact-name').value.trim();
      const email = document.getElementById('contact-email').value.trim();
      const message = document.getElementById('contact-message').value.trim();
      
      if (name && email && message) {
        const classifiedId = parseInt(contactClassifiedModal.dataset.classifiedId);
        const classified = API.getClassifiedById(classifiedId);
        
        if (classified.type === 'offer') {
          alert(`Has contratado con éxito: ${classified.title}`);
        } else {
          alert(`Has aplicado con éxito a: ${classified.title}`);
        }
        
        this.hideContactClassifiedModal();
      } else {
        alert('Por favor, completa todos los campos obligatorios.');
      }
    },
    
    // Cambiar tipo de servicio
    changeServiceType(type) {
      this.state.currentServiceType = type;
      
      // Actualizar botones
      const serviceTypeButtons = document.querySelectorAll('.service-type-btn');
      serviceTypeButtons.forEach(btn => {
        if (btn.dataset.type === type) {
          btn.classList.add('active');
        } else {
          btn.classList.remove('active');
        }
      });
      
      this.renderClassifieds();
    },
    
    // Cambiar filtro
    changeFilter(filter) {
      this.state.currentFilter = filter;
      
      // Actualizar botones de filtro
      const filterTags = document.querySelectorAll('.tag');
      filterTags.forEach(tag => {
        if (tag.dataset.filter === filter) {
          tag.classList.add('active');
        } else {
          tag.classList.remove('active');
        }
      });
      
      this.renderClassifieds();
    },
    
    // Buscar clasificados
    searchClassifieds(query) {
      this.state.searchQuery = query;
      this.renderClassifieds();
    },
    
    // Configurar eventos
    setupEventListeners() {
      // Cambiar entre tipos de servicio
      const serviceTypeButtons = document.querySelectorAll('.service-type-btn');
      serviceTypeButtons.forEach(btn => {
        btn.addEventListener('click', () => this.changeServiceType(btn.dataset.type));
      });
      
      // Filtros
      const filterTags = document.querySelectorAll('.tag');
      filterTags.forEach(tag => {
        tag.addEventListener('click', () => this.changeFilter(tag.dataset.filter));
      });
      
      // Búsqueda
      const searchInput = document.getElementById('search-input');
      searchInput.addEventListener('input', (e) => {
        this.searchClassifieds(e.target.value);
      });
      
      // Menú móvil
      const mobileMenuToggle = document.getElementById('mobile-menu-toggle');
      const mobileMenu = document.getElementById('mobile-menu');
      mobileMenuToggle.addEventListener('click', () => {
        mobileMenu.classList.toggle('active');
      });
      
      // Modal de crear clasificado
      const createClassifiedBtn = document.getElementById('create-classified-btn');
      const cancelClassifiedBtn = document.getElementById('cancel-classified-btn');
      const submitClassifiedBtn = document.getElementById('submit-classified-btn');
      
      createClassifiedBtn.addEventListener('click', () => this.showCreateClassifiedModal());
      cancelClassifiedBtn.addEventListener('click', () => this.hideCreateClassifiedModal());
      submitClassifiedBtn.addEventListener('click', () => this.createClassified());
      
      // Modal de contactar
      const cancelContactBtn = document.getElementById('cancel-contact-btn');
      const submitContactBtn = document.getElementById('submit-contact-btn');
      
      cancelContactBtn.addEventListener('click', () => this.hideContactClassifiedModal());
      submitContactBtn.addEventListener('click', () => this.submitContact());
      
      // Cerrar modales al hacer clic fuera
      window.addEventListener('click', e => {
        if (e.target.classList.contains('modal-backdrop')) {
          this.hideCreateClassifiedModal();
          this.hideContactClassifiedModal();
        }
      });
    }
  };
  
  // Iniciar la aplicación cuando el DOM esté cargado
  document.addEventListener('DOMContentLoaded', () => App.init());