import express from 'express';
import User from '../models/user.js'; // Asegúrate de tener un modelo de usuario definido
import { methods as authorization } from "../middlewares/authorization.js";
import Entidad from '../models/entidad.js'; // Asegúrate de que la ruta sea correcta

import dotenv from 'dotenv';

dotenv.config();
const router = express.Router();

// Ruta para obtener la cantidad de usuarios
router.get('/count', async (req, res) => {
    try {
      const userCount = await User.count(); // Método correcto en Sequelize
      res.status(200).json({ count: userCount });
    } catch (error) {
      console.error('Error al contar usuarios:', error);
      res.status(500).json({ error: 'Error al obtener la cantidad de usuarios' });
    }
  });
  // Ruta para obtener todos los usuarios
  router.get('/', async (req, res) => {
    try {
      const users = await User.findAll(); // Método para obtener todos los usuarios en Sequelize
      res.status(200).json(users);
    } catch (error) {
      console.error('Error al obtener usuarios:', error);
      res.status(500).json({ error: 'Error al obtener los usuarios' });
    }
  });

// Ruta DELETE para eliminar un usuario, solo accesible por admins
router.delete('/:id', authorization.soloAdmin, async (req, res) => {
  try {
    const userId = req.params.id; // ID del usuario a eliminar

    // Buscar el usuario en la base de datos
    const user = await User.findByPk(userId);

    // Si no se encuentra al usuario, responde con un error
    if (!user) {
      return res.status(404).json({ error: 'Usuario no encontrado' });
    }

    // Verificar si el usuario tiene entidades asociadas
    const entidadesAsociadas = await Entidad.findAll({ where: { UserAdminId: userId } });

    // Si tiene entidades asociadas, mostrar un mensaje y preguntar si eliminar
    if (entidadesAsociadas.length > 0) {
      return res.status(400).json({
        error: 'Este usuario tiene entidades asociadas. ¿Deseas eliminar también las entidades?',
        decision: 'Confirmar eliminación de usuario y entidades asociadas'
      });
    }

    // Si no tiene entidades asociadas, proceder a eliminar el usuario
    await user.destroy();
    res.status(200).json({ message: 'Usuario eliminado correctamente' });

  } catch (error) {
    console.error('Error al eliminar usuario:', error);
    res.status(500).json({ error: 'Error al eliminar el usuario' });
  }
});

router.delete('/:id/with-entidades', authorization.soloAdmin, async (req, res) => {
  try {
    const userId = req.params.id;

    // Buscar el usuario en la base de datos
    const user = await User.findByPk(userId);

    if (!user) {
      return res.status(404).json({ error: 'Usuario no encontrado' });
    }

    // Buscar y eliminar las entidades asociadas
    const entidadesAsociadas = await Entidad.findAll({ where: { UserAdminId: userId } });
    if (entidadesAsociadas.length > 0) {
      await Entidad.destroy({ where: { UserAdminId: userId } });
    }

    // Eliminar el usuario
    await user.destroy();
    res.status(200).json({ message: 'Usuario y entidades eliminados correctamente' });

  } catch (error) {
    console.error('Error al eliminar usuario y entidades:', error);
    res.status(500).json({ error: 'Error al eliminar el usuario y las entidades asociadas' });
  }
});

// Ruta para obtener solo el correo de un usuario por ID
router.get('/email/:id', async (req, res) => {
  try {
    const userId = req.params.id;

    // Buscar el usuario en la base de datos
    const user = await User.findByPk(userId, {
      attributes: ['email'] // Seleccionar solo el campo de correo
    });

    // Si no se encuentra al usuario, responde con un error
    if (!user) {
      return res.status(404).json({ error: 'Usuario no encontrado' });
    }

    // Responder con el correo del usuario
    res.status(200).json(user);

  } catch (error) {
    console.error('Error al obtener correo del usuario por ID:', error);
    res.status(500).json({ error: 'Error al obtener el correo del usuario' });
  }
});
export default router;